<?php

namespace App\Services;

use App\Models\Courier;
use App\Models\Delivery;
use App\Models\Restaurant;
use App\Models\WalletTransaction;
use Illuminate\Support\Facades\DB;

class WalletService
{
    public function creditRestaurant(Restaurant $restaurant, float $amount, ?string $description = null, ?int $deliveryId = null, ?int $createdBy = null): void
    {
        $this->adjustRestaurant($restaurant, $amount, 'credit', $description, $deliveryId, $createdBy);
    }

    public function debitRestaurant(Restaurant $restaurant, float $amount, ?string $description = null, ?int $deliveryId = null, ?int $createdBy = null): void
    {
        $this->adjustRestaurant($restaurant, -abs($amount), 'debit', $description, $deliveryId, $createdBy);
    }

    public function creditCourier(Courier $courier, float $amount, ?string $description = null, ?int $deliveryId = null, ?int $createdBy = null, string $type = 'credit'): void
    {
        $this->adjustCourier($courier, $amount, $type, $description, $deliveryId, $createdBy);
    }

    public function debitCourier(Courier $courier, float $amount, ?string $description = null, ?int $deliveryId = null, ?int $createdBy = null, string $type = 'debit'): void
    {
        $this->adjustCourier($courier, -abs($amount), $type, $description, $deliveryId, $createdBy);
    }

    public function transferDeliveryFee(Delivery $delivery, ?int $createdBy = null): void
    {
        if (!$delivery->courier || $delivery->delivery_fee <= 0) {
            return;
        }

        if ($delivery->fee_transferred_at) {
            return;
        }

        $this->creditCourier(
            $delivery->courier,
            $delivery->delivery_fee,
            'Pago de entrega #' . $delivery->id,
            $delivery->id,
            $createdBy,
            'transfer_in'
        );

        if ($delivery->payment_method === 'cash') {
            $this->recordRestaurantMovement(
                $delivery->restaurant,
                $delivery->delivery_fee,
                'cash_out',
                'Pago en efectivo entrega #' . $delivery->id,
                $delivery->id,
                $createdBy
            );
        }
    }

    protected function adjustRestaurant(Restaurant $restaurant, float $delta, string $type, ?string $description, ?int $deliveryId, ?int $createdBy): void
    {
        DB::transaction(function () use ($restaurant, $delta, $type, $description, $deliveryId, $createdBy) {
            $restaurant->refresh();
            $newBalance = round($restaurant->wallet_balance + $delta, 2);
            $restaurant->wallet_balance = $newBalance;
            $restaurant->save();

            WalletTransaction::create([
                'owner_type' => 'restaurant',
                'owner_id' => $restaurant->id,
                'type' => $type,
                'amount' => abs($delta),
                'balance_after' => $newBalance,
                'delivery_id' => $deliveryId,
                'created_by' => $createdBy,
                'description' => $description,
            ]);
        });
    }

    public function recordRestaurantMovement(Restaurant $restaurant, float $amount, string $type, ?string $description = null, ?int $deliveryId = null, ?int $createdBy = null): void
    {
        DB::transaction(function () use ($restaurant, $amount, $type, $description, $deliveryId, $createdBy) {
            $restaurant->refresh();
            WalletTransaction::create([
                'owner_type' => 'restaurant',
                'owner_id' => $restaurant->id,
                'type' => $type,
                'amount' => abs($amount),
                'balance_after' => $restaurant->wallet_balance,
                'delivery_id' => $deliveryId,
                'created_by' => $createdBy,
                'description' => $description,
            ]);
        });
    }

    protected function adjustCourier(Courier $courier, float $delta, string $type, ?string $description, ?int $deliveryId, ?int $createdBy): void
    {
        DB::transaction(function () use ($courier, $delta, $type, $description, $deliveryId, $createdBy) {
            $courier->refresh();
            $newBalance = round($courier->wallet_balance + $delta, 2);
            $courier->wallet_balance = $newBalance;
            $courier->save();

            WalletTransaction::create([
                'owner_type' => 'courier',
                'owner_id' => $courier->id,
                'type' => $type,
                'amount' => abs($delta),
                'balance_after' => $newBalance,
                'delivery_id' => $deliveryId,
                'created_by' => $createdBy,
                'description' => $description,
            ]);
        });
    }
}
